<?php

namespace App\Http\Controllers\Admin;

use App\User;
use App\Setting;
use App\Currency;
use App\Transaction;
use App\Classes\AvailableVersion;
use App\BusinessCard;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class DashboardController extends Controller
{
    /** 
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the application dashboard.
     *
     * @return \Illuminate\Contracts\Support\Renderable
     */
    public function index()
    {
        // Queries
        $settings = Setting::where('status', 1)->first();
        $config = DB::table('config')->get();
        $currency = Currency::where('iso_code', $config['1']->config_value)->first();
        $thisMonthIncome = Transaction::whereBetween('created_at', [Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth()])->where('payment_status', 'Success')->sum('transaction_amount');
        $today_income = Transaction::where('payment_status', 'Success')->whereDate('created_at', Carbon::today())->sum('transaction_amount');
        $overall_users = User::where('role_id', 2)->where('status', 1)->count();
        $today_users = User::where('role_id', 2)->where('status', 1)->whereDate('created_at', Carbon::today())->count();

        // Chart
        $year = date('Y');
        $monthIncome = Transaction::where('payment_status', 'Success')
            ->whereYear('created_at', $year)
            ->groupBy(DB::raw('MONTH(created_at)'))
            ->selectRaw('MONTH(created_at) as month, SUM(transaction_amount) as total_income')
            ->pluck('total_income', 'month')
            ->toArray();

        $monthUsers = User::where('role_id', 2)
            ->whereYear('created_at', $year)
            ->groupBy(DB::raw('MONTH(created_at)'))
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as total_users')
            ->pluck('total_users', 'month')
            ->toArray();

        $monthIncome = implode(',', array_pad($monthIncome, 12, 0));
        $monthUsers = implode(',', array_pad($monthUsers, 12, 0));

        // Overview chart
        $year = date('Y');

        // Earnings query
        $earnings = Transaction::where('payment_status', 'Success')
            ->whereYear('created_at', $year)
            ->groupBy(DB::raw('MONTH(created_at)'))
            ->selectRaw('MONTH(created_at) as month, SUM(transaction_amount) as total_earnings')
            ->pluck('total_earnings', 'month')
            ->toArray();

        // VCards and Store counts queries
        $vcards = BusinessCard::where('card_type', 'vcard')
            ->where('card_status', '!=', 'deleted')
            ->where('status', 1)
            ->whereYear('created_at', $year)
            ->groupBy(DB::raw('MONTH(created_at)'))
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as total_vcards')
            ->pluck('total_vcards', 'month')
            ->toArray();

        $stores = BusinessCard::where('card_type', 'store')
            ->where('card_status', '!=', 'deleted')
            ->where('status', 1)
            ->whereYear('created_at', $year)
            ->groupBy(DB::raw('MONTH(created_at)'))
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as total_stores')
            ->pluck('total_stores', 'month')
            ->toArray();

        // Pad missing months with 0s
        $earnings = implode(',', array_pad($earnings, 12, 0));
        $vcards = implode(',', array_pad($vcards, 12, 0));
        $stores = implode(',', array_pad($stores, 12, 0));

        // Total vCards and stores count
        $cardCounts = BusinessCard::where('card_status', '!=', 'deleted')
            ->where('status', 1)
            ->selectRaw('card_type, COUNT(*) as count')
            ->groupBy('card_type')
            ->pluck('count', 'card_type')
            ->toArray();

        $totalvCards = $cardCounts['vcard'] ?? 0;
        $totalStores = $cardCounts['store'] ?? 0;

        // Total Earnings
        $totalEarnings = Transaction::where('payment_status', 'Success')->sum('transaction_amount');

        // Current week sales
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = $startOfWeek->copy()->endOfWeek();

        $currentWeekSales = Transaction::where('payment_status', 'Success')
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->selectRaw('DAYOFWEEK(created_at) - 1 as day, SUM(transaction_amount) as daily_sales')
            ->groupBy(DB::raw('DAYOFWEEK(created_at) - 1')) // Group by the adjusted day of the week (0 = Sunday, 6 = Saturday)
            ->pluck('daily_sales', 'day')
            ->toArray();

        // Pad missing days with 0s (in case no sales on a specific day)
        $currentWeekSales = array_pad($currentWeekSales, 7, 0);

        // Default message
        $available = new AvailableVersion;
        $resp_data = $available->availableVersion();

        if ($resp_data) {
            if ($resp_data['status'] == true && $resp_data['update'] == true) {
                // Store success message in session
                session()->flash('message', trans('<a href="' . route("admin.check") . '" class="text-white">A new version is available! <span style="position: absolute; right: 7.5vh;">' . trans("Install") . '</span></a>'));
            }
        }

        // View
        return view('admin.dashboard', compact('thisMonthIncome', 'today_income', 'overall_users', 'today_users', 'currency', 'settings', 'monthIncome', 'monthUsers', 'earnings', 'vcards', 'stores', 'totalEarnings', 'totalvCards', 'totalStores', 'currentWeekSales'));
    }
}
